using System;
using System.Data;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using System.Reflection;
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.DlgSelectSpecificities;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{
		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///Antisera Worklist business layer object. For use in Unit Antigen Typing use case.
		///</summary>

		#endregion

	public class AntiseraWorklist : gov.va.med.vbecs.BOL.BaseBusinessObject, IRecordStatusCode
	{
		#region Variables
		private System.Data.DataTable _antiseraWorklistTable;
		private System.Data.DataRow _antiseraWorklistRow;
		private bool _automatedInstrument;
		private Common.RecordStatusCode _recordStatusCode;

		#endregion

		#region Constructors
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3074"> 
		///		<ExpectedInput>New unqiue identifier</ExpectedInput>
		///		<ExpectedOutput>Empty antisera worklist object initialized</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="227"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AntiseraWorklist class constructor
		/// </summary>
		/// <param name="antiseraWorklistGuid">Antisera Worklist unique identifier</param>
		public AntiseraWorklist(System.Guid antiseraWorklistGuid)
		{
			_antiseraWorklistTable = DAL.AntiseraWorklist.GetAntiseraWorklistByGuid(antiseraWorklistGuid);
			if (_antiseraWorklistTable.Rows.Count > 0)
			{
				_antiseraWorklistRow = _antiseraWorklistTable.Rows[0];
				this.AutomatedInstrument = (this.AutoInstrumentName.Length>0);
				this.RowVersion = (byte[])_antiseraWorklistRow[TABLE.AntiseraWorklist.RowVersion];
			}
			else
			{
				MakeNew(antiseraWorklistGuid);
			}
			_antiseraWorklistRow.AcceptChanges();
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3076"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3077"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets Antisera Worklist unique identifier property
		/// </summary>
		public Guid AntiseraWorklistGuid
		{
			get
			{
				return (Guid)this._antiseraWorklistRow[TABLE.AntiseraWorklist.AntiseraWorklistGuid];
			}
			set
			{
				this._antiseraWorklistRow[TABLE.AntiseraWorklist.AntiseraWorklistGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3078"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3079"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>Broken Rule added</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets name of person that created the antisera worklist
		/// </summary>
		public string WorklistName
		{
			get
			{
				return this._antiseraWorklistRow[TABLE.AntiseraWorklist.WorklistName].ToString();
			}
			set
			{
				this._antiseraWorklistRow[TABLE.AntiseraWorklist.WorklistName] = value;
				//
				RuleBroken(MethodBase.GetCurrentMethod().Name,(value.Length==0));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="228"> 
		///		<ExpectedInput>Common.RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>Common.RecordStatusCode</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="229"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3080"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3081"> 
		///		<ExpectedInput>Future datetime</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3113"> 
		///		<ExpectedInput>DateTime.Minvalue</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets the date/time that the antisera worklist was created
		/// </summary>
		public DateTime WorklistCreateDate
		{
			get
			{
				return Convert.ToDateTime(this._antiseraWorklistRow[TABLE.AntiseraWorklist.WorklistCreateDate]);
			}
			set
			{
				if (value > VBECSDateTime.GetDivisionCurrentDateTime())
				{
					throw new BusinessObjectException("Future date not permitted");
				}
				this._antiseraWorklistRow[TABLE.AntiseraWorklist.WorklistCreateDate] = value;

				RuleBroken(MethodBase.GetCurrentMethod().Name,((value==DateTime.MinValue)||
																(value>VBECSDateTime.GetDivisionCurrentDateTime())));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="230"> 
		///		<ExpectedInput>Common.TestingMethod </ExpectedInput>
		///		<ExpectedOutput>Common.TestingMethod </ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="231"> 
		///		<ExpectedInput>Unknown testing method code</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// TestingMethodCode
		/// </summary>
		public Common.TestingMethod TestingMethodCode
		{
			get
			{
				return Common.Utility.GetTestingMethodEnumFromCode(Convert.ToChar(this._antiseraWorklistRow[TABLE.AntiseraWorklist.TestingMethodCode]));
			}
			set
			{
				this._antiseraWorklistRow[TABLE.AntiseraWorklist.TestingMethodCode] = Common.Utility.GetTestingMethodCodeFromEnum(value);
				RuleBroken(MethodBase.GetCurrentMethod().Name,(value==Common.TestingMethod.Unknown));
			}
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="232"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="233"> 
		///		<ExpectedInput>False</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// AutomatedInstrument
		/// </summary>
		public bool AutomatedInstrument
		{
			get
			{
				return this._automatedInstrument;
			}
			set
			{
				this._automatedInstrument = value;
				RuleBroken(BROKEN.AutoInstrumentName,value);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="234"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="235"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// AutoInstrumentName
		/// </summary>
		public string AutoInstrumentName
		{
			get
			{
				return this._antiseraWorklistRow[TABLE.AntiseraWorklist.AutoInstrumentName].ToString();
			}
			set
			{
				this._antiseraWorklistRow[TABLE.AntiseraWorklist.AutoInstrumentName] = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name,(value.Length==0));
			}
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///
		/// <summary>
		/// Private method to initialize a new instance of antisera worklist
		/// </summary>
		/// <param name="antiseraWorklistGuid">Antisera Worklist unique identifier</param>
		private void MakeNew(Guid antiseraWorklistGuid)
		{
			this.IsNew = true;
			this.RecordStatusCode = Common.RecordStatusCode.Active;
			this.AutomatedInstrument = false;
			//
			_antiseraWorklistRow = _antiseraWorklistTable.NewRow();
			_antiseraWorklistRow[TABLE.AntiseraWorklist.AntiseraWorklistGuid] = antiseraWorklistGuid;
			_antiseraWorklistRow[TABLE.AntiseraWorklist.WorklistCreateDate] = DateTime.MinValue;
			_antiseraWorklistRow[TABLE.AntiseraWorklist.WorklistName] = string.Empty;
			_antiseraWorklistRow[TABLE.AntiseraWorklist.AutoInstrumentName] = string.Empty;
			_antiseraWorklistRow[TABLE.AntiseraWorklist.TestingMethodCode] = Common.Utility.GetTestingMethodCodeFromEnum(Common.TestingMethod.Unknown);
			_antiseraWorklistRow[TABLE.AntiseraWorklist.LastUpdateUser] = Common.LogonUser.LogonUserName;
			_antiseraWorklistRow[TABLE.AntiseraWorklist.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(this.RecordStatusCode);
			_antiseraWorklistRow[TABLE.AntiseraWorklist.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			_antiseraWorklistTable.Rows.Add(_antiseraWorklistRow);
			
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3082"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of partially completed worklists</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3083"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of antisera worklists that have not been completed based on division code
		/// </summary>
		/// <returns>Antisera worklist data table</returns>
		public static DataTable GetPartiallyCompletedWorklists()
		{
			return DAL.AntiseraWorklist.GetPartiallyCompletedAntiseraWorklists();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3084"> 
		///		<ExpectedInput>Valid antisera worklist guid</ExpectedInput>
		///		<ExpectedOutput>Data table of antisera worklist details</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="236"> 
		///		<ExpectedInput>Invalid antisera worklist guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of antisera specificities in an antisera worklist based on the antisera worklist GUID
		/// </summary>
		/// <param name="antiseraWorklistGuid">Antisera worklist unique identifier</param>
		/// <returns>Rack Lot data table</returns>
		public static DataTable GetAntiseraWorklistDetails(Guid antiseraWorklistGuid)
		{
			return DAL.AntiseraWorklist.GetAntiseraWorklistDetails(antiseraWorklistGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3088"> 
		///		<ExpectedInput>Valid antisera worklist guid</ExpectedInput>
		///		<ExpectedOutput>Data table of antisera worklist units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="238"> 
		///		<ExpectedInput>Invalid antisera worklist guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of blood units in an antisera worklist based on the antisera worklist GUID
		/// </summary>
		/// <param name="antiseraWorklistGuid">Antisera worklist unique identifier</param>
		/// <returns>Blood Unit data table</returns>
		public static DataTable GetAntiseraWorklistUnits(Guid antiseraWorklistGuid)
		{
			return DAL.AntiseraWorklist.GetAntiseraWorklistUnits(antiseraWorklistGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3094"> 
		///		<ExpectedInput>Valid blood unit guid and antisera worklist typing guid</ExpectedInput>
		///		<ExpectedOutput>Data table of antisera worklist test results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="239"> 
		///		<ExpectedInput>Invalid blood unit guid and antisera worklist typing guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns Antigen Typing test results for a given blood unit and orderable tesr
		/// </summary>
		/// <param name="bloodUnitGuid">Blood unit unique identifier</param>
		/// <param name="antiseraWorklistTypingGuid">Antisera worklist typing unique identifier</param>
		/// <returns>Blood unit test data table</returns>
		public static DataTable GetTestResultsForAntiseraWorklist(Guid bloodUnitGuid, Guid antiseraWorklistTypingGuid)
		{
			return DAL.AntiseraWorklist.GetTestResultsForAntiseraWorklist(bloodUnitGuid, antiseraWorklistTypingGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3096"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Antisera worklist typing table schema</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="240"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get empty table schema for use when inserting antisera worklist typings into the database
		/// </summary>
		/// <returns>Antisera Worklist Typing table schema</returns>
		public static DataTable GetAntiseraWorklistTypingTableSchema()
		{
			DataTable dt = new DataTable(TABLE.AntiseraWorklistTyping.TableName);

			dt.Columns.Add(TABLE.AntiseraUnitTest.AntiseraWorklistTypingGuid, typeof(Guid));
			dt.Columns.Add(TABLE.AntiseraWorklist.AntiseraWorklistGuid, typeof(Guid));
			dt.Columns.Add(TABLE.OrderableTest.OrderableTestId,typeof(string));
			return dt;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3098"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Antisera unit test table schema</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="241"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get empty table schema for use when inserting antisera unit tests into the database
		/// </summary>
		/// <returns>Antisera Unit Test table schema</returns>
		public static DataTable GetAntiseraUnitTestTableSchema()
		{
			DataTable dt = new DataTable(TABLE.AntiseraUnitTest.TableName);

			dt.Columns.Add(TABLE.AntiseraUnitTest.AntiseraUnitTestGuid, typeof(Guid));
			dt.Columns.Add(TABLE.AntiseraUnitTest.AntiseraWorklistTypingGuid, typeof(Guid));
			dt.Columns.Add(TABLE.AntiseraUnitTest.RackLotGuid, typeof(Guid));				
			return dt;
		}

		private void AddRackLotRow(ref System.Data.DataRow drNewRow, System.Data.DataRow drRackLot, System.Data.DataRow drWorkList, bool controlCell)
		{
			drNewRow[TABLE.RackLot.RackLotGuid] = Guid.NewGuid();
			drNewRow[TABLE.RackLot.ReagentGuid] = drRackLot[TABLE.RackLot.ReagentGuid];
			drNewRow[TABLE.RackLot.ReactivityPhaseCode] = drRackLot[TABLE.RackLot.ReactivityPhaseCode];
			drNewRow[TABLE.RackLot.DivisionCode] = drWorkList[TABLE.AntiseraWorklist.DivisionCode];
			drNewRow[TABLE.RackLot.LotUseDate] = drWorkList[TABLE.AntiseraWorklist.WorklistCreateDate];
			drNewRow[TABLE.RackLot.RackGuid] = drRackLot[TABLE.RackLot.RackGuid];
			drNewRow[TABLE.RackLot.ControlCellId] = controlCell;
		}

		private void AddRackTestRow(ref System.Data.DataRow drNewRow, System.Data.DataRow drRackLot, System.Data.DataRow drWorkList, string rackTestResultTypeCode)
		{
			drNewRow[TABLE.RackTest.RackTestGuid] = Guid.NewGuid();
			drNewRow[TABLE.RackTest.RackLotGuid] = drRackLot[TABLE.RackLot.RackLotGuid];
			drNewRow[TABLE.RackTest.TestDate] = drWorkList[TABLE.AntiseraWorklist.WorklistCreateDate];
			drNewRow[TABLE.RackTest.RackTestResultTypeCode] = rackTestResultTypeCode;
			drNewRow[TABLE.RackTest.RackTestResult] = string.Empty;
			drNewRow[TABLE.RackTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			drNewRow[TABLE.RackTest.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drNewRow[TABLE.RackTest.TestTechId] = drWorkList[TABLE.AntiseraWorklist.WorklistName].ToString();
		}

		private void AddAntiseraRow(ref System.Data.DataRow drNewRow,System.Data.DataRow drTyping, System.Data.DataRow drWorkList, System.Data.DataRow drRackLot)
		{
			drNewRow[TABLE.AntiseraUnitTest.AntiseraUnitTestGuid] = Guid.NewGuid();
			drNewRow[TABLE.AntiseraUnitTest.AntiseraWorklistTypingGuid] = drTyping[TABLE.AntiseraUnitTest.AntiseraWorklistTypingGuid];
			drNewRow[TABLE.AntiseraUnitTest.RackLotGuid] = drRackLot[TABLE.RackLot.RackLotGuid];
		}

		private void AddAntiseraWorklistRow(ref System.Data.DataRow drNewRow,System.Data.DataRow drTyping, System.Data.DataRow drWorkList)
		{
			drNewRow[TABLE.AntiseraWorklist.AntiseraWorklistGuid] = drWorkList[TABLE.AntiseraWorklist.AntiseraWorklistGuid];
			drNewRow[TABLE.AntiseraUnitTest.AntiseraWorklistTypingGuid] = drTyping[TABLE.AntiseraUnitTest.AntiseraWorklistTypingGuid];
			drNewRow[TABLE.OrderableTest.OrderableTestId] = drTyping[TABLE.OrderableTest.OrderableTestId];
		}

		private void AddBloodUnitTest(ref System.Data.DataRow drNewRow,System.Data.DataRow drTyping,System.Data.DataRow drWorklist, System.Data.DataRow drTest, System.Data.DataRow drBloodUnit, char entryMethodCode )
		{
			drNewRow[TABLE.BloodUnitTest.BloodUnitTestGuid] = System.Guid.NewGuid();
			drNewRow[TABLE.AntiseraUnitTest.AntiseraWorklistTypingGuid] = drTyping[TABLE.AntiseraUnitTest.AntiseraWorklistTypingGuid];
			drNewRow[TABLE.BloodUnitTest.BloodUnitGuid] = drBloodUnit[TABLE.BloodUnit.BloodUnitGuid];
			drNewRow[TABLE.BloodUnitTest.BloodTestTypeId] = drTest[TABLE.BloodTestType.BloodTestTypeId];
			drNewRow[TABLE.BloodUnitTest.TestDate] = drWorklist[TABLE.AntiseraWorklist.WorklistCreateDate];
			drNewRow[TABLE.BloodUnitTest.TestResultId] = DBNull.Value;
			drNewRow[TABLE.BloodUnitTest.EntryMethodCode] = entryMethodCode;
			drNewRow[TABLE.BloodUnitTest.TestingMethodCode] = Common.Utility.GetTestingMethodCodeFromEnum(this.TestingMethodCode);
			drNewRow[TABLE.BloodUnitTest.TestComments] = DBNull.Value;
			drNewRow[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			drNewRow[TABLE.BloodUnitTest.RackGuid] = drTyping[TABLE.RackLot.RackGuid];
			drNewRow[TABLE.BloodUnitTest.AutoInstrumentName] = this.AutoInstrumentName;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="242"> 
		///		<ExpectedInput>Valid Antigen Typing data</ExpectedInput>
		///		<ExpectedOutput>Saved indicator of true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6827"> 
		///		<ExpectedInput>Valid Weak D Typing data</ExpectedInput>
		///		<ExpectedOutput>Saved indicator of true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="243"> 
		///		<ExpectedInput>Invalid antigen typing data (Reagent Guid not set)</ExpectedInput>
		///		<ExpectedOutput>Format Exception thrown</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Inserts empty antisera worklist in database. System creates a single AntiseraWorklist
		/// entry, several AntiseraWorklistTyping entries, several AntiseraUnitTest entries, several
		/// RackLot entries, several BloodUnitTest entries, and optionally several RackTest entries.
		/// Implements BR_41.03, BR_41.27
		/// </summary>
		/// <param name="antiseraWorklistTypingTable">Antisera Worklist Typing data table</param>
		/// <param name="bloodUnitTable">Blood Unit data table</param>
		/// <param name="testingTechName">Name of the testing tech</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="bloodUnitGuids"></param>
		/// <param name="antiDControlGuid">Weak D Anti-D Control reagent guid (optional)</param>
		/// <param name="antiDControlComment">Weak D Anti-D Control exception comment (optional)</param>
		/// <param name="partiallyTested"></param>
		/// <returns>Success indicator</returns>
		public bool Save(DataTable antiseraWorklistTypingTable, 
			DataTable bloodUnitTable, 
			string testingTechName, 
			Common.UpdateFunction lastUpdateFunctionId,
			System.Guid [] bloodUnitGuids,
			Guid antiDControlGuid,
			string antiDControlComment,
			bool partiallyTested)
		{
			DataSet dsSave = new DataSet();
			bool includeDCtrl = false;
			DataRow drWorklist = _antiseraWorklistTable.Rows[0];
			//
			_antiseraWorklistTable.TableName = TABLE.AntiseraWorklist.TableName;
			dsSave.Tables.Add(_antiseraWorklistTable.Copy());
			//
			DataTable antiseraWorklistTypings = BOL.AntiseraWorklist.GetAntiseraWorklistTypingTableSchema();
			dsSave.Tables.Add(antiseraWorklistTypings);
			//
			DataTable rackLotTable = BOL.RackLot.GetRackLotTableSchema(true);
			dsSave.Tables.Add(rackLotTable);
			//
			DataTable antiseraUnitTestTable = BOL.AntiseraWorklist.GetAntiseraUnitTestTableSchema();
			dsSave.Tables.Add(antiseraUnitTestTable);
			//
			DataTable rackTestTable = BOL.RackTest.GetRackTestTableSchema(true);
			dsSave.Tables.Add(rackTestTable);
			//
			DataTable bloodUnitTestTable = BOL.BloodUnitTest.GetBloodUnitTestTableSchema(false);
			dsSave.Tables.Add(bloodUnitTestTable);
			//
			DataTable dtExceptionExpiredAntisera = null;
			DataTable dtExceptionPhaseChange = null;
			//
			DataTable dtTests = BOL.AntigenTyping.GetAntigenTypingTests();
			//Need to set this, because by default it is false, and Select comparisons will be case-insensitive
			dtTests.CaseSensitive = true;

			foreach (DataRow drTyping in antiseraWorklistTypingTable.Rows)
			{
				if (drTyping[TABLE.RackLot.RackGuid].ToString() == Guid.Empty.ToString())
				{
					drTyping[TABLE.RackLot.RackGuid] = System.DBNull.Value;
				}
				////Need to save exceptions here..
				if (drTyping.IsNull(ARTIFICIAL.ExpiredOverrideComment) == false)
				{
					//Need to generate an exception for each blood unit involved
					for (int idx = 0; idx < bloodUnitGuids.Length; idx++)
					{
						Guid reagentGuid = new Guid(drTyping[TABLE.RackLot.ReagentGuid].ToString());
						BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Expired_antisera_used, drTyping[ARTIFICIAL.ExpiredOverrideComment].ToString(), Common.LogonUser.LogonUserName);
						//
						if (drTyping.IsNull(TABLE.RackLot.RackGuid))
						{
							exRep.GenerateExceptionQC(reagentGuid, Guid.Empty, bloodUnitGuids[idx], Guid.Empty, "", "", Common.Utility.GetReactivityPhaseTextFromCode(drTyping[TABLE.RackLot.ReactivityPhaseCode].ToString()), testingTechName);
						}
						else
						{
							Guid rackGuid = new Guid(drTyping[TABLE.RackLot.RackGuid].ToString());
							exRep.GenerateExceptionQC(reagentGuid, rackGuid, bloodUnitGuids[idx], Guid.Empty, "", "", Common.Utility.GetReactivityPhaseTextFromCode(drTyping[TABLE.RackLot.ReactivityPhaseCode].ToString()), testingTechName);
						}
						//
						if (dtExceptionExpiredAntisera == null)
						{
							dtExceptionExpiredAntisera = exRep.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionExpiredAntisera can't be null since it is assigned earler (see above). 
                        dtExceptionExpiredAntisera.Rows.Add(exRep.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }
                }
				//
				if (drTyping.IsNull(ARTIFICIAL.PhaseOverrideComment) == false)
				{
					string orderableTestName = "";
					foreach(DataRow drTest in dtTests.Rows)
					{
						if (drTest[TABLE.OrderableTest.OrderableTestId].ToString() == drTyping[TABLE.OrderableTest.OrderableTestId].ToString())
						{
							orderableTestName = drTest[TABLE.OrderableTest.OrderableTestName].ToString();
							break;
						}
					}
					//
					foreach(DataRow drUnit in bloodUnitTable.Rows)
					{
						Guid unitGuid = new Guid(drUnit[TABLE.BloodUnit.BloodUnitGuid].ToString());
						BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Antigen_testing_phase_change, drTyping[ARTIFICIAL.PhaseOverrideComment].ToString(), Common.LogonUser.LogonUserName);
						exRep.GenerateExceptionUnitTest(unitGuid, drUnit[TABLE.BloodUnit.EyeReadableUnitId].ToString(),orderableTestName,Common.Utility.GetReactivityPhaseTextFromCode(drTyping[TABLE.RackLot.ReactivityPhaseCode].ToString()),"",Convert.ToDateTime(drWorklist[TABLE.AntiseraWorklist.WorklistCreateDate]),testingTechName  );
						if (dtExceptionPhaseChange == null)
						{
							dtExceptionPhaseChange = exRep.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionPhaseChange can't be null since it is assigned earler (see above). 
                        dtExceptionPhaseChange.Rows.Add(exRep.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }
                }

				DataRow drAWT = antiseraWorklistTypings.NewRow();
				this.AddAntiseraWorklistRow(ref drAWT, drTyping, drWorklist);
				antiseraWorklistTypings.Rows.Add(drAWT);
				//
				string phase = Common.Utility.GetReactivityPhaseTextFromCode(drTyping[TABLE.RackLot.ReactivityPhaseCode].ToString());
				DataTable bloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(drTyping[TABLE.OrderableTest.OrderableTestId].ToString());
				//
				if (Convert.ToInt32(drTyping[TABLE.OrderableTest.OrderableTestId]) == (int)Common.OrderableTest.AGwD)
				{
					//Check if D-Control part of daily QC
					includeDCtrl = true;
					
					//Anti D Control RackLot
					DataRow drDcRackLot = rackLotTable.NewRow();
					this.AddRackLotRow(ref drDcRackLot, drTyping, drWorklist, false);
					drDcRackLot[TABLE.RackLot.ReagentGuid] = antiDControlGuid;
					rackLotTable.Rows.Add(drDcRackLot);
					//
					//Anti D Control AntiseraUnitTest
					DataRow drDcAUT = antiseraUnitTestTable.NewRow();
					this.AddAntiseraRow(ref drDcAUT, drTyping, drWorklist, drDcRackLot);
					antiseraUnitTestTable.Rows.Add(drDcAUT);
					//
					if (antiDControlComment != null)
					{
						BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Expired_antisera_used, antiDControlComment, Common.LogonUser.LogonUserName);
						//
						//Need to generate an exception for each blood unit involved
						for (int idx = 0; idx < bloodUnitGuids.Length; idx++)
						{
							exRep.GenerateExceptionQC(antiDControlGuid, Guid.Empty, bloodUnitGuids[idx], Guid.Empty, "", "", Common.Utility.GetReactivityPhaseTextFromCode(((int)Common.ReactivityPhase.ThreeSevenandAHG).ToString()), testingTechName);
						}
						//
						if (dtExceptionExpiredAntisera == null)
						{
							dtExceptionExpiredAntisera = exRep.ExceptionData.Table.Clone();
						}
                        //*** Fortify Justified Code ***
                        //dtExceptionExpiredAntisera can't be null since it is assigned earler (see above). 
                        dtExceptionExpiredAntisera.Rows.Add(exRep.ExceptionData.ItemArray);
                        //*** Fortify Justified Code ***
                    }
                }
				else
				{
					//BR_41.03, BR_41.27
					//Check if QC'd today using same testing phase
					//If no QC Results, add POS and NEG control for Reagent to worklist
					DataTable dtReagentTestResults = BOL.RackTest.GetReagentTestResult(Common.LogonUser.LogonUserDivisionCode, new Guid(drTyping[TABLE.RackLot.ReagentGuid].ToString()), Convert.ToDateTime(drWorklist[TABLE.AntiseraWorklist.WorklistCreateDate]),Convert.ToChar(drTyping[TABLE.RackLot.ReactivityPhaseCode].ToString()));
					if (dtReagentTestResults.Rows.Count==0 || 
						dtReagentTestResults.Select(Common.VbecsTables.RackLot.SatisfactoryResultIndicator + " IS NULL").Length > 0) //CR 1574 Added check to see if the QC results were completed
					{
						//Pos RackLot
						DataRow drPosQCRackLot = rackLotTable.NewRow();
						this.AddRackLotRow(ref drPosQCRackLot, drTyping, drWorklist, true);
						rackLotTable.Rows.Add(drPosQCRackLot);
						//
						//Neg RackLot
						DataRow drNegQCRackLot = rackLotTable.NewRow();
						this.AddRackLotRow(ref drNegQCRackLot, drTyping, drWorklist, false);
						rackLotTable.Rows.Add(drNegQCRackLot);
						//
						//Pos AntiseraUnitTest
						DataRow drPos = antiseraUnitTestTable.NewRow();
						AddAntiseraRow(ref drPos, drTyping, drWorklist, drPosQCRackLot);
						antiseraUnitTestTable.Rows.Add(drPos);
						//
						//Neg AntiseraUnitTest
						DataRow drNeg = antiseraUnitTestTable.NewRow();
						AddAntiseraRow(ref drNeg, drTyping, drWorklist, drNegQCRackLot);
						antiseraUnitTestTable.Rows.Add(drNeg);
						//
						if (phase.IndexOf(ARTIFICIAL.IS)>-1)
						{
							DataRow drPosRackTest = rackTestTable.NewRow();
							this.AddRackTestRow(ref drPosRackTest, drPosQCRackLot, drWorklist, ARTIFICIAL.IS);
							rackTestTable.Rows.Add(drPosRackTest);
							//
							DataRow drNegRackTest = rackTestTable.NewRow();
							this.AddRackTestRow(ref drNegRackTest, drNegQCRackLot, drWorklist, ARTIFICIAL.IS);
							rackTestTable.Rows.Add(drNegRackTest);
						}
						if (phase.IndexOf(ARTIFICIAL.RT)>-1)
						{
							DataRow drPosRackTest = rackTestTable.NewRow();
							this.AddRackTestRow(ref drPosRackTest, drPosQCRackLot, drWorklist, ARTIFICIAL.RT);
							rackTestTable.Rows.Add(drPosRackTest);
							//
							DataRow drNegRackTest = rackTestTable.NewRow();
							this.AddRackTestRow(ref drNegRackTest, drNegQCRackLot, drWorklist, ARTIFICIAL.RT);
							rackTestTable.Rows.Add(drNegRackTest);
						}
						if (phase.IndexOf(ARTIFICIAL.C37)>-1)
						{
							//
							DataRow drPosRackTest = rackTestTable.NewRow();
							this.AddRackTestRow(ref drPosRackTest, drPosQCRackLot, drWorklist, ARTIFICIAL.C37);
							rackTestTable.Rows.Add(drPosRackTest);
							//
							DataRow drNegRackTest = rackTestTable.NewRow();
							this.AddRackTestRow(ref drNegRackTest, drNegQCRackLot, drWorklist, ARTIFICIAL.C37);								
							rackTestTable.Rows.Add(drNegRackTest);
						}
						if (phase.IndexOf(ARTIFICIAL.AHG)>-1)
						{
							//
							DataRow drPosAHG = rackTestTable.NewRow();
							this.AddRackTestRow(ref drPosAHG, drPosQCRackLot, drWorklist, ARTIFICIAL.AHG);								
							rackTestTable.Rows.Add(drPosAHG);
							//
							DataRow drNegAHG = rackTestTable.NewRow();
							this.AddRackTestRow(ref drNegAHG, drNegQCRackLot, drWorklist, ARTIFICIAL.AHG);								
							rackTestTable.Rows.Add(drNegAHG);
							//
							DataRow drPosCC = rackTestTable.NewRow();
							this.AddRackTestRow(ref drPosCC, drPosQCRackLot, drWorklist, ARTIFICIAL.CC);																
							rackTestTable.Rows.Add(drPosCC);
							//
							DataRow drNegCC = rackTestTable.NewRow();
							this.AddRackTestRow(ref drNegCC, drNegQCRackLot, drWorklist, ARTIFICIAL.CC);																
							rackTestTable.Rows.Add(drNegCC);
						}
					}
				}
				//RackLot
				DataRow drRackLot = rackLotTable.NewRow();
				this.AddRackLotRow(ref drRackLot, drTyping, drWorklist, false);
				drRackLot[TABLE.RackLot.ControlCellId] = System.DBNull.Value;
				rackLotTable.Rows.Add(drRackLot);
				
				//AntiseraUnitTest
				DataRow drAUT = antiseraUnitTestTable.NewRow();
				this.AddAntiseraRow(ref drAUT, drTyping, drWorklist, drRackLot);
				
				antiseraUnitTestTable.Rows.Add(drAUT);
				//
				foreach(DataRow drTest in bloodTestTypes.Rows)
				{
					if (Convert.ToInt32(drTyping[TABLE.OrderableTest.OrderableTestId]) == (int)Common.OrderableTest.AGwD)
					{
						if ((drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("wD 37")>-1 && (phase.IndexOf(ARTIFICIAL.C37)>-1))||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("wD AHG")>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("wD CC")>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("D Control 37")>-1 && (phase.IndexOf(ARTIFICIAL.C37)>-1) && includeDCtrl)||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("D Control AHG")>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1) && includeDCtrl)||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("D Control CC")>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1) && includeDCtrl)||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.Interp)>-1))
						{
							foreach (DataRow drBloodUnit in bloodUnitTable.Rows)
							{
								//
								DataRow drBUT = bloodUnitTestTable.NewRow();
								this.AddBloodUnitTest(ref drBUT, drTyping, drWorklist, drTest,drBloodUnit, 'M');
								bloodUnitTestTable.Rows.Add(drBUT);
							}
						}
					}
					else
					{
						if ((drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.IS)>-1 && (phase.IndexOf(ARTIFICIAL.IS)>-1))||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.RT)>-1 && (phase.IndexOf(ARTIFICIAL.RT)>-1))||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.C37)>-1 && (phase.IndexOf(ARTIFICIAL.C37)>-1))||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.AHG)>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.CC)>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
							(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.Interp)>-1))
						{
							foreach (DataRow drBloodUnit in bloodUnitTable.Rows)
							{
								//
								DataRow drBUT = bloodUnitTestTable.NewRow();
								this.AddBloodUnitTest(ref drBUT, drTyping, drWorklist, drTest,drBloodUnit, 'M');
								bloodUnitTestTable.Rows.Add(drBUT);
							}
						}
					}
				}
			}
			if (dtExceptionExpiredAntisera != null)
				dsSave.Tables.Add(dtExceptionExpiredAntisera);
			if (dtExceptionPhaseChange != null)
				dsSave.Tables.Add(dtExceptionPhaseChange);

			bool retval = DAL.AntiseraWorklist.Save(dsSave, lastUpdateFunctionId);
			if (retval) 
			{
				_antiseraWorklistTable.AcceptChanges();
				return true;
			}
			return false;
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3102"> 
		///		<ExpectedInput>Valid antisera worklist</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3103"> 
		///		<ExpectedInput>Antisera worklist with invalid row version</ExpectedInput>
		///		<ExpectedOutput>Row Version exception thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates the antisera worklist task status. This can be used to indicate that a worklist has been completed
		/// </summary>
		/// <returns>Success indicator</returns>
		public bool UpdateTaskStatus(Common.TaskStatus taskStatus)
		{
			return DAL.AntiseraWorklist.UpdateTaskStatus(this.AntiseraWorklistGuid,
									Common.Utility.GetTaskStatusCodeFromEnum(taskStatus),
									this.RowVersion);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3104"> 
		///		<ExpectedInput>Valid antisera worklist typing guid</ExpectedInput>
		///		<ExpectedOutput>Data table of antisera worklist qc results</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="244"> 
		///		<ExpectedInput>Invalid antisera worklist typing guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets Daily QC Results for a given antigen typing reagent
		/// </summary>
		/// <param name="antiseraWorklistTypingGuid">Antisera worklist typing unqiue identifier</param>
		/// <returns>Rack test data table</returns>
		public static DataTable GetAntiseraWorklistQCResults(Guid antiseraWorklistTypingGuid)
		{
			return DAL.AntiseraWorklist.GetAntiseraWorklistQCResults(antiseraWorklistTypingGuid);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3106"> 
		///		<ExpectedInput>Valid antisera worklist typing Guid</ExpectedInput>
		///		<ExpectedOutput>Data table containing QC control info</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="245"> 
		///		<ExpectedInput>Invalid antisera worklist typing guid</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the positive and negative control rack lot information for a given antigen typing
		/// </summary>
		/// <param name="antiseraWorklistTypingGuid">Antisera worklist typing unique identifier</param>
		/// <returns>Rack Lot data table</returns>
		public static DataTable GetAntiseraWorklistQCControls(Guid antiseraWorklistTypingGuid)
		{
			return DAL.AntiseraWorklist.GetAntiseraWorklistQCControls(antiseraWorklistTypingGuid);
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>Unknown</CreationDate>
		///<TestCases>
		/// 
		///<Case type="0" testid ="7352"> 
		///		<ExpectedInput>Valid antisera worklist, worklist typing table, unit table and comment</ExpectedInput>
		///		<ExpectedOutput>Success indicator = true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7353"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InvalidateAntiseraWorklist
		/// </summary>
		/// <param name="antiseraWorklistTypings"></param>
		/// <param name="bloodUnits"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <param name="overrideComment"></param>
		/// <param name="updateTaskStatus"></param>
		/// <returns></returns>
		public bool InvalidateAntiseraWorklist(DataTable antiseraWorklistTypings, DataTable bloodUnits, Common.UpdateFunction lastUpdateFunctionId, string overrideComment, bool updateTaskStatus)
		{
			DataTable dtAntiseraWorklist = null;
			//
			if (updateTaskStatus)
			{
				dtAntiseraWorklist = new DataTable(TABLE.AntiseraWorklist.TableName);
				dtAntiseraWorklist.Columns.Add(TABLE.AntiseraWorklist.AntiseraWorklistGuid,typeof(Guid));
				dtAntiseraWorklist.Columns.Add(TABLE.AntiseraWorklist.RecordStatusCode,typeof(char));
				dtAntiseraWorklist.Columns.Add(TABLE.AntiseraWorklist.LastUpdateUser,typeof(string));
				dtAntiseraWorklist.Columns.Add(TABLE.AntiseraWorklist.LastUpdateFunctionId,typeof(int));
				dtAntiseraWorklist.Columns.Add(TABLE.AntiseraWorklist.RowVersion,typeof(byte[]));
				//
				DataRow drAntiseraWorklist = dtAntiseraWorklist.NewRow();
				drAntiseraWorklist[TABLE.AntiseraWorklist.AntiseraWorklistGuid] = this.AntiseraWorklistGuid;
				drAntiseraWorklist[TABLE.AntiseraWorklist.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
				drAntiseraWorklist[TABLE.AntiseraWorklist.LastUpdateUser] = Common.LogonUser.LogonUserName;
				drAntiseraWorklist[TABLE.AntiseraWorklist.LastUpdateFunctionId] = (int)lastUpdateFunctionId;
				drAntiseraWorklist[TABLE.AntiseraWorklist.RowVersion] = this.RowVersion;
				dtAntiseraWorklist.Rows.Add(drAntiseraWorklist);
			}
			//
			DataTable dtExceptions = null;
			DataTable dtAntigenTypingGuids = new DataTable();;
			dtAntigenTypingGuids.Columns.Add(TABLE.AntiseraWorklistTyping.AntiseraWorklistTypingGuid,typeof(Guid));
			//
			foreach(DataRow antiseraWorklistTyping in antiseraWorklistTypings.Rows)
			{
				foreach(DataRow drUnit in bloodUnits.Rows)
				{
					DataTable unitTestResults = BOL.AntiseraWorklist.GetTestResultsForAntiseraWorklist((Guid)drUnit[TABLE.BloodUnitTest.BloodUnitGuid],(Guid)antiseraWorklistTyping[TABLE.AntiseraWorklistTyping.AntiseraWorklistTypingGuid]);
					//
					foreach(DataRow bloodUnitTest in unitTestResults.Rows)
					{
						//BR_41.28 - Generate exception for invalidated test result
						if (bloodUnitTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(" Interp")>-1)
						{
							BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Previously_recorded_antigen_results_invalidated, overrideComment, Common.LogonUser.LogonUserName);
							exRep.GenerateExceptionAntigenTest(bloodUnitTest, this.WorklistName, this.WorklistCreateDate);
							//
							if (dtExceptions == null)
							{
								dtExceptions = exRep.ExceptionData.Table.Clone();
							}
                            //*** Fortify Justified Code ***
                            //dtExceptions can't be null since it is assigned earler (see above). 
                            dtExceptions.Rows.Add(exRep.ExceptionData.ItemArray);
                            //*** Fortify Justified Code ***
                        }
                    }
				}
				//
				DataRow drTyping = dtAntigenTypingGuids.NewRow();
				drTyping[TABLE.AntiseraWorklistTyping.AntiseraWorklistTypingGuid] = (Guid)antiseraWorklistTyping[TABLE.AntiseraWorklistTyping.AntiseraWorklistTypingGuid];
				dtAntigenTypingGuids.Rows.Add(drTyping);
			}
			return (DAL.AntiseraWorklist.InvalidateAntigenTypingTestResults(dtAntigenTypingGuids, dtAntiseraWorklist,lastUpdateFunctionId, dtExceptions));

		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="246"> 
		///		<ExpectedInput>Valid antiseraWorklistTyping datarow, bloodUnits datatable and lastUpdateFunctionId for AHG CC phases</ExpectedInput>
		///		<ExpectedOutput>True bool value</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="247"> 
		///		<ExpectedInput>Invalid datarow, datatable</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7106"> 
		///		<ExpectedInput>Valid antiseraWorklistTyping datarow, bloodUnits datatable and lastUpdateFunctionId for IS RT phases</ExpectedInput>
		///		<ExpectedOutput>True bool value</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7107"> 
		///		<ExpectedInput>Valid antiseraWorklistTyping datarow, bloodUnits datatable and lastUpdateFunctionId for IS 37 phases</ExpectedInput>
		///		<ExpectedOutput>True bool value</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invalidates existing antigen typing test results and creates new empty replacement test results
		/// </summary>
		/// <param name="antiseraWorklistTyping">Antisera worklist typing data row</param>
		/// <param name="bloodUnits">Blood units data table</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="overrideComment">Override comment</param>
		/// <returns>Success indicator</returns>
		public bool InvalidateAntigenTypingTestResults(DataRow antiseraWorklistTyping, DataTable bloodUnits, Common.UpdateFunction lastUpdateFunctionId, string overrideComment)
		{
			bool retval = false;
			//
			if (this.InvalidateAntiseraWorklist(antiseraWorklistTyping.Table, bloodUnits, lastUpdateFunctionId, overrideComment, false))
			{
				DataRow drWorklist = _antiseraWorklistTable.Rows[0];
				//
				DataTable antiseraUnitTestTable = BOL.AntiseraWorklist.GetAntiseraUnitTestTableSchema();
				//
				DataTable bloodUnitTestTable = BOL.BloodUnitTest.GetBloodUnitTestTableSchema(false);
				//
				DataTable rackLotTable = BOL.RackLot.GetRackLotTableSchema(true);
				//
				DataTable rackTestTable = BOL.RackTest.GetRackTestTableSchema(true);
				//
				string phase = Common.Utility.GetReactivityPhaseTextFromCode(antiseraWorklistTyping[TABLE.RackLot.ReactivityPhaseCode].ToString());
				DataTable bloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(antiseraWorklistTyping[TABLE.OrderableTest.OrderableTestId].ToString());
				//
				//Unit Tests
				if (antiseraWorklistTyping[TABLE.RackLot.RackGuid].ToString() == Guid.Empty.ToString())
				{
					antiseraWorklistTyping[TABLE.RackLot.RackGuid] = System.DBNull.Value;
				}
				//
				//Check if QC'd today
				//If no QC Results, add POS and NEG control for Reagent to worklist
				DataTable dtReagentTestResults = BOL.RackTest.GetReagentTestResult(Common.LogonUser.LogonUserDivisionCode, (Guid)antiseraWorklistTyping[TABLE.RackLot.ReagentGuid]);
				if (dtReagentTestResults.Rows.Count==0)
				{
					//Pos RackLot
					DataRow drPosQCRackLot = rackLotTable.NewRow();
					this.AddRackLotRow(ref drPosQCRackLot,antiseraWorklistTyping, drWorklist, true );
					rackLotTable.Rows.Add(drPosQCRackLot);
					//
					//Neg RackLot
					DataRow drNegQCRackLot = rackLotTable.NewRow();
					this.AddRackLotRow(ref drNegQCRackLot, antiseraWorklistTyping, drWorklist, false);
					rackLotTable.Rows.Add(drNegQCRackLot);
					//
					//Pos AntiseraUnitTest
					DataRow drPos = antiseraUnitTestTable.NewRow();					
					this.AddAntiseraRow(ref drPos, antiseraWorklistTyping, drWorklist, drPosQCRackLot);
					antiseraUnitTestTable.Rows.Add(drPos);
					//
					//Neg AntiseraUnitTest
					DataRow drNeg = antiseraUnitTestTable.NewRow();
					this.AddAntiseraRow(ref drNeg, antiseraWorklistTyping, drWorklist, drNegQCRackLot);
					antiseraUnitTestTable.Rows.Add(drNeg);
					//
					if (phase.IndexOf(ARTIFICIAL.IS)>-1)
					{
						DataRow drPosRackTest = rackTestTable.NewRow();
						this.AddRackTestRow(ref drPosRackTest, drPosQCRackLot, drWorklist, ARTIFICIAL.IS);
						rackTestTable.Rows.Add(drPosRackTest);
						//
						DataRow drNegRackTest = rackTestTable.NewRow();
						this.AddRackTestRow(ref drNegRackTest, drNegQCRackLot, drWorklist, ARTIFICIAL.IS);
						rackTestTable.Rows.Add(drNegRackTest);
					}
					if (phase.IndexOf(ARTIFICIAL.RT)>-1)
					{
						DataRow drPosRackTest = rackTestTable.NewRow();
						this.AddRackTestRow(ref drPosRackTest, drPosQCRackLot, drWorklist, ARTIFICIAL.RT);
						rackTestTable.Rows.Add(drPosRackTest);
						//
						DataRow drNegRackTest = rackTestTable.NewRow();
						this.AddRackTestRow(ref drNegRackTest, drNegQCRackLot, drWorklist, ARTIFICIAL.RT);
						rackTestTable.Rows.Add(drNegRackTest);
					}
					if (phase.IndexOf(ARTIFICIAL.C37)>-1)
					{
						//
						DataRow drPosRackTest = rackTestTable.NewRow();
						this.AddRackTestRow(ref drPosRackTest, drPosQCRackLot, drWorklist, ARTIFICIAL.C37);
						rackTestTable.Rows.Add(drPosRackTest);
						//
						DataRow drNegRackTest = rackTestTable.NewRow();
						this.AddRackTestRow(ref drNegRackTest, drNegQCRackLot,drWorklist, ARTIFICIAL.C37); 
						rackTestTable.Rows.Add(drNegRackTest);
					}
					if (phase.IndexOf(ARTIFICIAL.AHG)>-1)
					{
						//
						DataRow drPosAHG = rackTestTable.NewRow();
						this.AddRackTestRow(ref drPosAHG, drPosQCRackLot, drWorklist, ARTIFICIAL.AHG);
						rackTestTable.Rows.Add(drPosAHG);
						//
						DataRow drNegAHG = rackTestTable.NewRow();
						this.AddRackTestRow(ref drNegAHG, drNegQCRackLot, drWorklist, ARTIFICIAL.AHG);
						rackTestTable.Rows.Add(drNegAHG);
						//
						DataRow drPosCC = rackTestTable.NewRow();
						this.AddRackTestRow(ref drPosCC, drPosQCRackLot, drWorklist, ARTIFICIAL.CC);
						rackTestTable.Rows.Add(drPosCC);
						//
						DataRow drNegCC = rackTestTable.NewRow();
						this.AddRackTestRow(ref drNegCC, drNegQCRackLot, drWorklist, ARTIFICIAL.CC);
						rackTestTable.Rows.Add(drNegCC);
					}
				}
				//RackLot
				DataRow drRackLot = rackLotTable.NewRow();
				this.AddRackLotRow(ref drRackLot, antiseraWorklistTyping, drWorklist, false);
				drRackLot[TABLE.RackLot.ControlCellId] = DBNull.Value;
                rackLotTable.Rows.Add(drRackLot);
				//
				//AntiseraUnitTest
				DataRow drAUT = antiseraUnitTestTable.NewRow();
				this.AddAntiseraRow(ref drAUT, antiseraWorklistTyping, drWorklist, drRackLot);
				antiseraUnitTestTable.Rows.Add(drAUT);
				//
				foreach(DataRow drTest in bloodTestTypes.Rows)
				{
					if ((drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.IS)>-1 && (phase.IndexOf(ARTIFICIAL.IS)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.RT)>-1 && (phase.IndexOf(ARTIFICIAL.RT)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.C37)>-1 && (phase.IndexOf(ARTIFICIAL.C37)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.AHG)>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.CC)>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.Interp)>-1))
					{
						foreach (DataRow drBloodUnit in bloodUnits.Rows)
						{
							//
							DataRow drBUT = bloodUnitTestTable.NewRow();
							this.AddBloodUnitTest(ref drBUT,  antiseraWorklistTyping, drWorklist, drTest, drBloodUnit, 'M' );
							bloodUnitTestTable.Rows.Add(drBUT);
						}
					}
				}
				//
				retval = DAL.AntiseraWorklist.Save(rackLotTable,antiseraUnitTestTable,rackTestTable,bloodUnitTestTable, lastUpdateFunctionId);
				if (retval) 
				{
					_antiseraWorklistTable.AcceptChanges();
				}
			}
			return retval;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="248"> 
		///		<ExpectedInput>Valid antiseraWorklistTyping datarow, bloodUnits datatable and lastUpdateFunctionId</ExpectedInput>
		///		<ExpectedOutput>True bool value</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="249"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invalidates existing Weak D antigen typing test results and creates new empty replacement test results
		/// </summary>
		/// <param name="antiseraWorklistTyping">Weak D antisera worklist typing data row</param>
		/// <param name="bloodUnits">Blood units data table</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="overrideComment">Override comment</param>
		/// <returns>Success indicator</returns>
		public bool InvalidateWeakDTestResults(DataRow antiseraWorklistTyping, DataTable bloodUnits, Common.UpdateFunction lastUpdateFunctionId, string overrideComment)
		{
			bool retval = false;
			//
			//CR 2881 Getting the DControl information from the database
			DataTable dtControls = BOL.AntiseraWorklist.GetAntiseraWorklistQCControls((Guid)antiseraWorklistTyping[TABLE.AntiseraWorklistTyping.AntiseraWorklistTypingGuid]);
			DataRow[] drDControl = dtControls.Select(Common.VbecsTables.ReagentType.ReagentTypeId + " = 53");
			
			if (this.InvalidateAntiseraWorklist(antiseraWorklistTyping.Table, bloodUnits, lastUpdateFunctionId, overrideComment, false))
			{
				DataRow drWorklist = _antiseraWorklistTable.Rows[0];
				//
				DataTable antiseraUnitTestTable = BOL.AntiseraWorklist.GetAntiseraUnitTestTableSchema();
				//
				DataTable bloodUnitTestTable = BOL.BloodUnitTest.GetBloodUnitTestTableSchema(false);
				//
				DataTable rackLotTable = BOL.RackLot.GetRackLotTableSchema(true);
				//
				DataTable rackTestTable = BOL.RackTest.GetRackTestTableSchema(true);
				//
				string phase = Common.Utility.GetReactivityPhaseTextFromCode(antiseraWorklistTyping[TABLE.RackLot.ReactivityPhaseCode].ToString());
				DataTable bloodTestTypes = BOL.BloodTestType.GetBloodTestTypes(antiseraWorklistTyping[TABLE.OrderableTest.OrderableTestId].ToString());
				//
				//Anti D RackLot
				DataRow drRackLot = rackLotTable.NewRow();
				this.AddRackLotRow(ref drRackLot, antiseraWorklistTyping, drWorklist, false);
				drRackLot[TABLE.RackLot.ControlCellId] = DBNull.Value;
				rackLotTable.Rows.Add(drRackLot);
				//
				//Anti D AntiseraUnitTest
				DataRow drAUT = antiseraUnitTestTable.NewRow();
				this.AddAntiseraRow(ref drAUT, antiseraWorklistTyping, drWorklist, drRackLot);
				antiseraUnitTestTable.Rows.Add(drAUT);
				//

				//CR 2881 Added code creating new RackLot based on the data in the worklist
				if (drDControl.Length > 0)
				{
					DataRow antiseraWorklistTypingForDControl = antiseraWorklistTyping.Table.NewRow();
					antiseraWorklistTypingForDControl.ItemArray = antiseraWorklistTyping.ItemArray;
					antiseraWorklistTypingForDControl[Common.VbecsTables.Reagent.ReagentGuid] = drDControl[0][Common.VbecsTables.Reagent.ReagentGuid];
					DataRow drDcRackLot = rackLotTable.NewRow();
					this.AddRackLotRow(ref drDcRackLot, antiseraWorklistTypingForDControl, drWorklist, false);
					rackLotTable.Rows.Add(drDcRackLot);
					//
					//Anti D Control AntiseraUnitTest
					DataRow drDcAUT = antiseraUnitTestTable.NewRow();
					this.AddAntiseraRow(ref drDcAUT, antiseraWorklistTyping, drWorklist, drDcRackLot);
					antiseraUnitTestTable.Rows.Add(drDcAUT);
				}

				//
				//Unit Tests
				foreach(DataRow drTest in bloodTestTypes.Rows)
				{
					if ((drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("wD 37")>-1 && (phase.IndexOf(ARTIFICIAL.C37)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("wD AHG")>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("wD CC")>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("D Control 37")>-1 && (phase.IndexOf(ARTIFICIAL.C37)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("D Control AHG")>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf("D Control CC")>-1 && (phase.IndexOf(ARTIFICIAL.AHG)>-1))||
						(drTest[TABLE.BloodTestType.BloodTestName].ToString().IndexOf(ARTIFICIAL.Interp)>-1))
					{
						foreach (DataRow drBloodUnit in bloodUnits.Rows)
						{
							//
							DataRow drBUT = bloodUnitTestTable.NewRow();
							this.AddBloodUnitTest(ref drBUT, antiseraWorklistTyping, drWorklist, drTest, drBloodUnit, 'M');
							bloodUnitTestTable.Rows.Add(drBUT);
						}
					}
				}
				//
				retval = DAL.AntiseraWorklist.Save(rackLotTable,antiseraUnitTestTable,rackTestTable,bloodUnitTestTable, lastUpdateFunctionId);
				if (retval) 
				{
					_antiseraWorklistTable = DAL.AntiseraWorklist.GetAntiseraWorklistByGuid(this.AntiseraWorklistGuid);
					_antiseraWorklistRow = _antiseraWorklistTable.Rows[0];
					this.AutomatedInstrument = (this.AutoInstrumentName.Length>0);
					this.RowVersion = (byte[])_antiseraWorklistRow[TABLE.AntiseraWorklist.RowVersion];
					_antiseraWorklistRow.AcceptChanges();
					_antiseraWorklistTable.AcceptChanges();
				}
			}
			return retval;
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="254"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="7357"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Business Object Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object - ** NOT IMPLEMENTED **
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dataRow)
		{
			throw new BusinessObjectException();
		}


		///
		/// <summary>
		/// Loads an object from a data row - ** NOT IMPLEMENTED **
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			//Not implemented
			throw new BusinessObjectException();
		}


        #endregion
	}
}
